/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.cert.ocsp.test;

import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.Provider;
import java.security.Security;
import java.util.Date;
import java.util.Random;
import java.util.Set;
import java.util.Vector;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Exception;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.ExtensionsGenerator;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cert.ocsp.CertificateID;
import org.bouncycastle.cert.ocsp.OCSPReq;
import org.bouncycastle.cert.ocsp.OCSPReqBuilder;
import org.bouncycastle.cert.ocsp.OCSPResp;
import org.bouncycastle.cert.ocsp.OCSPRespBuilder;
import org.bouncycastle.cert.ocsp.Req;
import org.bouncycastle.cert.ocsp.RespID;
import org.bouncycastle.cert.ocsp.SingleResp;
import org.bouncycastle.cert.ocsp.jcajce.JcaBasicOCSPRespBuilder;
import org.bouncycastle.cert.ocsp.test.OCSPTestUtil;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.DigestCalculatorProvider;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.operator.jcajce.JcaDigestCalculatorProviderBuilder;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.test.SimpleTest;
import org.bouncycastle.util.test.Test;

public class OCSPTest
extends SimpleTest {
    byte[] testResp1 = Base64.decode((String)"MIIFnAoBAKCCBZUwggWRBgkrBgEFBQcwAQEEggWCMIIFfjCCARehgZ8wgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4YDzIwMDMwNDAyMTIzNDU4WjBiMGAwOjAJBgUrDgMCGgUABBRs07IuoCWNmcEl1oHwIak1BPnX8QQUtGyl/iL9WJ1VxjxFj0hAwJ/s1AcCAQKhERgPMjAwMjA4MjkwNzA5MjZaGA8yMDAzMDQwMjEyMzQ1OFowDQYJKoZIhvcNAQEFBQADgYEAfbN0TCRFKdhsmvOdUoiJ+qvygGBzDxD/VWhXYA+16AphHLIWNABR3CgHB3zWtdy2j7DJmQ/R7qKj7dUhWLSqclAiPgFtQQ1YvSJAYfEIdyHkxv4NP0LSogxrumANcDyC9yt/W9yHjD2ICPBIqCsZLuLkOHYi5DlwWe9Zm9VFwCGgggPMMIIDyDCCA8QwggKsoAMCAQICAQYwDQYJKoZIhvcNAQEFBQAwgZQxFDASBgNVBAMTC1RDUy1DQSBPQ1NQMSYwJAYJKoZIhvcNAQkBFhd0Y3MtY2FAdGNzLWNhLnRjcy5jby5pbjEMMAoGA1UEChMDVENTMQwwCgYDVQQLEwNBVEMxEjAQBgNVBAcTCUh5ZGVyYWJhZDEXMBUGA1UECBMOQW5kaHJhIHByYWRlc2gxCzAJBgNVBAYTAklOMB4XDTAyMDgyOTA3MTE0M1oXDTAzMDgyOTA3MTE0M1owgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAM+XWW4caMRv46D7L6Bv8iwtKgmQu0SAybmFRJiz12qXzdvTLt8C75OdgmUomxp0+gW/4XlTPUqOMQWv463aZRv9Ust4f8MHEJh4ekP/NS9+d8vEO3P40ntQkmSMcFmtA9E1koUtQ3MSJlcs441JjbgUaVnmjDmmniQnZY4bU3tVAgMBAAGjgZowgZcwDAYDVR0TAQH/BAIwADALBgNVHQ8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwkwNgYIKwYBBQUHAQEEKjAoMCYGCCsGAQUFBzABhhpodHRwOi8vMTcyLjE5LjQwLjExMDo3NzAwLzAtBgNVHR8EJjAkMCKgIKAehhxodHRwOi8vMTcyLjE5LjQwLjExMC9jcmwuY3JsMA0GCSqGSIb3DQEBBQUAA4IBAQB6FovM3B4VDDZ15o12gnADZsIk9fTAczLlcrmXLNN4PgmqgnwF0Ymj3bD5SavDOXxbA65AZJ7rBNAguLUo+xVkgxmoBH7R2sBxjTCcr07NEadxM3HQkt0aX5XYEl8eRoifwqYAI9h0ziZfTNes8elNfb3DoPPjqq6VmMg0f0iMS4W8LjNPorjRB+kIosa1deAGPhq0eJ8yr0/s2QR2/WFD5P4aXc8IKWleklnIImS3zqiPrq6tl2Bm8DZj7vXlTOwmraSQxUwzCKwYob1yGvNOUQTqpG6jxn7jgDawHU1+WjWQe4Q34/pWeGLysxTraMa+Ug9kPe+jy/qRX2xwvKBZ");
    byte[] testResp2 = Base64.decode((String)"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");
    private static byte[] irregReq = Base64.decode((String)"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");
    private static byte[] invalidResp = Base64.decode((String)"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");
    private final byte[] emptyExtResp = Base64.decode((String)"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");
    private static final String BC = "BC";

    public String getName() {
        return "OCSP";
    }

    public static void main(String[] stringArray) throws Exception {
        Security.addProvider((Provider)new BouncyCastleProvider());
        SimpleTest.runTest((Test)new OCSPTest());
    }

    public void performTest() throws Exception {
        block26: {
            OCSPReq oCSPReq;
            block25: {
                OCSPResp oCSPResp;
                Extension extension;
                ASN1Encodable aSN1Encodable;
                Set set;
                byte[] byArray;
                OCSPReq oCSPReq2;
                Req[] reqArray;
                X509CertificateHolder[] x509CertificateHolderArray;
                String string = "O=Bouncy Castle, C=AU";
                KeyPair keyPair = OCSPTestUtil.makeKeyPair();
                JcaX509CertificateHolder jcaX509CertificateHolder = new JcaX509CertificateHolder(OCSPTestUtil.makeCertificate(keyPair, string, keyPair, string));
                String string2 = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
                GeneralName generalName = new GeneralName(new X500Name(string2));
                DigestCalculatorProvider digestCalculatorProvider = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
                CertificateID certificateID = new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L));
                OCSPReqBuilder oCSPReqBuilder = new OCSPReqBuilder();
                oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
                oCSPReq = oCSPReqBuilder.build();
                if (oCSPReq.isSigned()) {
                    this.fail("signed but shouldn't be");
                }
                if ((x509CertificateHolderArray = oCSPReq.getCerts()).length != 0) {
                    this.fail("0 certs expected, but not found");
                }
                if (!(reqArray = oCSPReq.getRequestList())[0].getCertID().equals(certificateID)) {
                    this.fail("Failed isFor test");
                }
                X509CertificateHolder[] x509CertificateHolderArray2 = new X509CertificateHolder[1];
                oCSPReqBuilder = new OCSPReqBuilder();
                oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
                oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
                x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
                oCSPReq = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
                if (!oCSPReq.isSigned()) {
                    this.fail("not signed but should be");
                }
                if (!oCSPReq.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
                    this.fail("signature failed to verify");
                }
                if (!(reqArray = oCSPReq.getRequestList())[0].getCertID().equals(certificateID)) {
                    this.fail("Failed isFor test");
                }
                if ((x509CertificateHolderArray = oCSPReq.getCerts()) == null) {
                    this.fail("null certs found");
                }
                if (x509CertificateHolderArray.length != 1 || !x509CertificateHolderArray[0].equals(jcaX509CertificateHolder)) {
                    this.fail("incorrect certs found in request");
                }
                if (!(oCSPReq2 = new OCSPReq(byArray = oCSPReq.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
                    this.fail("newReq signature failed to verify");
                }
                x509CertificateHolderArray2 = new X509CertificateHolder[1];
                oCSPReqBuilder = new OCSPReqBuilder();
                Vector vector = new Vector();
                Vector vector2 = new Vector();
                byte[] byArray2 = new byte[16];
                Random random = new Random();
                random.nextBytes(byArray2);
                oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
                ExtensionsGenerator extensionsGenerator = new ExtensionsGenerator();
                extensionsGenerator.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, (ASN1Encodable)new DEROctetString(byArray2));
                oCSPReqBuilder.setRequestExtensions(extensionsGenerator.generate());
                oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
                x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
                oCSPReq = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
                if (!oCSPReq.isSigned()) {
                    this.fail("not signed but should be");
                }
                if (!oCSPReq.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
                    this.fail("signature failed to verify");
                }
                if ((set = oCSPReq.getCriticalExtensionOIDs()).size() != 0) {
                    this.fail("wrong number of critical extensions in OCSP request.");
                }
                if ((set = oCSPReq.getNonCriticalExtensionOIDs()).size() != 1) {
                    this.fail("wrong number of non-critical extensions in OCSP request.");
                }
                if (!((aSN1Encodable = (extension = oCSPReq.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
                    this.fail("wrong extension type found.");
                }
                if (!this.areEqual(((ASN1OctetString)aSN1Encodable).getOctets(), byArray2)) {
                    this.fail("wrong extension value found.");
                }
                OCSPResp oCSPResp2 = new OCSPResp(this.emptyExtResp);
                BasicOCSPResp basicOCSPResp = (BasicOCSPResp)oCSPResp2.getResponseObject();
                this.isTrue(basicOCSPResp.hasExtensions());
                this.isEquals(0, basicOCSPResp.getExtensionOIDs().size());
                reqArray = oCSPReq.getRequestList();
                if (!reqArray[0].getCertID().equals(certificateID)) {
                    this.fail("Failed isFor test");
                }
                if ((oCSPResp = new OCSPResp(this.testResp1)).getStatus() != 0) {
                    this.fail("response status not zero.");
                }
                BasicOCSPResp basicOCSPResp2 = (BasicOCSPResp)oCSPResp.getResponseObject();
                x509CertificateHolderArray2 = basicOCSPResp2.getCerts();
                if (!basicOCSPResp2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(x509CertificateHolderArray2[0]))) {
                    this.fail("response 1 failed to verify.");
                }
                SingleResp[] singleRespArray = basicOCSPResp2.getResponses();
                oCSPResp = new OCSPResp(this.testResp2);
                if (oCSPResp.getStatus() != 0) {
                    this.fail("response status not zero.");
                }
                basicOCSPResp2 = (BasicOCSPResp)oCSPResp.getResponseObject();
                x509CertificateHolderArray2 = basicOCSPResp2.getCerts();
                if (!basicOCSPResp2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(x509CertificateHolderArray2[0]))) {
                    this.fail("response 2 failed to verify.");
                }
                singleRespArray = basicOCSPResp2.getResponses();
                OCSPRespBuilder oCSPRespBuilder = new OCSPRespBuilder();
                OCSPResp oCSPResp3 = oCSPRespBuilder.build(0, oCSPResp.getResponseObject());
                if (!oCSPResp3.getResponseObject().equals(oCSPResp.getResponseObject())) {
                    this.fail("response fails to match");
                }
                this.testECDSA();
                this.testRSA();
                this.testIrregularVersionReq();
                this.testInvalidResp();
                try {
                    oCSPResp = new OCSPResp(new byte[0]);
                    this.fail("no exception thrown");
                }
                catch (IOException iOException) {
                    if (iOException.getMessage().equals("malformed response: no response data found")) break block25;
                    this.fail("wrong exception");
                }
            }
            try {
                oCSPReq = new OCSPReq(new byte[0]);
                this.fail("no exception thrown");
            }
            catch (IOException iOException) {
                if (iOException.getMessage().equals("malformed request: no request data found")) break block26;
                this.fail("wrong exception");
            }
        }
    }

    private void testECDSA() throws Exception {
        Extension extension;
        ASN1Encodable aSN1Encodable;
        Set set;
        byte[] byArray;
        OCSPReq oCSPReq;
        Req[] reqArray;
        X509CertificateHolder[] x509CertificateHolderArray;
        String string = "O=Bouncy Castle, C=AU";
        KeyPair keyPair = OCSPTestUtil.makeECKeyPair();
        JcaX509CertificateHolder jcaX509CertificateHolder = new JcaX509CertificateHolder(OCSPTestUtil.makeECDSACertificate(keyPair, string, keyPair, string));
        DigestCalculatorProvider digestCalculatorProvider = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
        String string2 = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName generalName = new GeneralName(new X500Name(string2));
        CertificateID certificateID = new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L));
        OCSPReqBuilder oCSPReqBuilder = new OCSPReqBuilder();
        oCSPReqBuilder.addRequest(certificateID);
        OCSPReq oCSPReq2 = oCSPReqBuilder.build();
        if (oCSPReq2.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((x509CertificateHolderArray = oCSPReq2.getCerts()).length != 0) {
            this.fail("0 certs expected, but not found");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        X509CertificateHolder[] x509CertificateHolderArray2 = new X509CertificateHolder[1];
        oCSPReqBuilder = new OCSPReqBuilder();
        oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
        oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
        x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
        oCSPReq2 = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
        if (!oCSPReq2.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!oCSPReq2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        if ((x509CertificateHolderArray = oCSPReq2.getCerts()) == null) {
            this.fail("null certs found");
        }
        if (x509CertificateHolderArray.length != 1 || !x509CertificateHolderArray[0].equals(jcaX509CertificateHolder)) {
            this.fail("incorrect certs found in request");
        }
        if (!(oCSPReq = new OCSPReq(byArray = oCSPReq2.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("newReq signature failed to verify");
        }
        x509CertificateHolderArray2 = new X509CertificateHolder[1];
        oCSPReqBuilder = new OCSPReqBuilder();
        Vector vector = new Vector();
        Vector vector2 = new Vector();
        byte[] byArray2 = new byte[16];
        Random random = new Random();
        random.nextBytes(byArray2);
        oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
        ExtensionsGenerator extensionsGenerator = new ExtensionsGenerator();
        extensionsGenerator.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, (ASN1Encodable)new DEROctetString(byArray2));
        oCSPReqBuilder.setRequestExtensions(extensionsGenerator.generate());
        oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
        x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
        oCSPReq2 = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
        if (!oCSPReq2.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!oCSPReq2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if ((set = oCSPReq2.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((set = oCSPReq2.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((aSN1Encodable = (extension = oCSPReq2.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)aSN1Encodable).getOctets(), byArray2)) {
            this.fail("wrong extension value found.");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        JcaBasicOCSPRespBuilder jcaBasicOCSPRespBuilder = new JcaBasicOCSPRespBuilder(keyPair.getPublic(), digestCalculatorProvider.get(RespID.HASH_SHA1));
        jcaBasicOCSPRespBuilder.addResponse(certificateID, null);
        BasicOCSPResp basicOCSPResp = jcaBasicOCSPRespBuilder.build(new JcaContentSignerBuilder("SHA1withECDSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2, new Date());
    }

    public void testInvalidResp() throws Exception {
        try {
            OCSPResp oCSPResp = new OCSPResp(invalidResp);
        }
        catch (CertIOException certIOException) {
            if (certIOException.getCause() instanceof ASN1Exception) {
                Throwable throwable = ((ASN1Exception)certIOException.getCause()).getCause();
                if (!throwable.getMessage().equals("ENUMERATED has zero length")) {
                    this.fail("parsing failed, but for wrong reason: " + throwable.getMessage());
                }
            }
            this.fail("parsing failed, but for wrong reason: " + certIOException.getMessage());
        }
    }

    private void testIrregularVersionReq() throws Exception {
        OCSPReq oCSPReq = new OCSPReq(irregReq);
        X509CertificateHolder x509CertificateHolder = oCSPReq.getCerts()[0];
        if (!oCSPReq.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(x509CertificateHolder))) {
            this.fail("extra version encoding test failed");
        }
    }

    private void testRSA() throws Exception {
        Extension extension;
        ASN1Encodable aSN1Encodable;
        Set set;
        byte[] byArray;
        OCSPReq oCSPReq;
        Req[] reqArray;
        X509CertificateHolder[] x509CertificateHolderArray;
        String string = "O=Bouncy Castle, C=AU";
        KeyPair keyPair = OCSPTestUtil.makeKeyPair();
        JcaX509CertificateHolder jcaX509CertificateHolder = new JcaX509CertificateHolder(OCSPTestUtil.makeCertificate(keyPair, string, keyPair, string));
        DigestCalculatorProvider digestCalculatorProvider = new JcaDigestCalculatorProviderBuilder().setProvider(BC).build();
        String string2 = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName generalName = new GeneralName(new X500Name(string2));
        CertificateID certificateID = new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L));
        OCSPReqBuilder oCSPReqBuilder = new OCSPReqBuilder();
        oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
        OCSPReq oCSPReq2 = oCSPReqBuilder.build();
        if (oCSPReq2.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((x509CertificateHolderArray = oCSPReq2.getCerts()).length != 0) {
            this.fail("0 certs expected, but not found");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        X509CertificateHolder[] x509CertificateHolderArray2 = new X509CertificateHolder[1];
        oCSPReqBuilder = new OCSPReqBuilder();
        oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
        oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
        x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
        oCSPReq2 = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
        if (!oCSPReq2.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!oCSPReq2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        if ((x509CertificateHolderArray = oCSPReq2.getCerts()) == null) {
            this.fail("null certs found");
        }
        if (x509CertificateHolderArray.length != 1 || !x509CertificateHolderArray[0].equals(jcaX509CertificateHolder)) {
            this.fail("incorrect certs found in request");
        }
        if (!(oCSPReq = new OCSPReq(byArray = oCSPReq2.getEncoded())).isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("newReq signature failed to verify");
        }
        x509CertificateHolderArray2 = new X509CertificateHolder[1];
        oCSPReqBuilder = new OCSPReqBuilder();
        byte[] byArray2 = new byte[16];
        Random random = new Random();
        random.nextBytes(byArray2);
        oCSPReqBuilder.setRequestorName(new GeneralName(4, (ASN1Encodable)new X500Name("CN=fred")));
        ExtensionsGenerator extensionsGenerator = new ExtensionsGenerator();
        extensionsGenerator.addExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce, false, (ASN1Encodable)new DEROctetString(byArray2));
        oCSPReqBuilder.setRequestExtensions(extensionsGenerator.generate());
        oCSPReqBuilder.addRequest(new CertificateID(digestCalculatorProvider.get(CertificateID.HASH_SHA1), jcaX509CertificateHolder, BigInteger.valueOf(1L)));
        x509CertificateHolderArray2[0] = jcaX509CertificateHolder;
        oCSPReq2 = oCSPReqBuilder.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2);
        if (!oCSPReq2.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!oCSPReq2.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider(BC).build(keyPair.getPublic()))) {
            this.fail("signature failed to verify");
        }
        if ((set = oCSPReq2.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((set = oCSPReq2.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((aSN1Encodable = (extension = oCSPReq2.getExtension(OCSPObjectIdentifiers.id_pkix_ocsp_nonce)).getParsedValue()) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)aSN1Encodable).getOctets(), byArray2)) {
            this.fail("wrong extension value found.");
        }
        if (!(reqArray = oCSPReq2.getRequestList())[0].getCertID().equals(certificateID)) {
            this.fail("Failed isFor test");
        }
        JcaBasicOCSPRespBuilder jcaBasicOCSPRespBuilder = new JcaBasicOCSPRespBuilder(keyPair.getPublic(), digestCalculatorProvider.get(RespID.HASH_SHA1));
        jcaBasicOCSPRespBuilder.addResponse(certificateID, null);
        BasicOCSPResp basicOCSPResp = jcaBasicOCSPRespBuilder.build(new JcaContentSignerBuilder("SHA1withRSA").setProvider(BC).build(keyPair.getPrivate()), x509CertificateHolderArray2, new Date());
        OCSPRespBuilder oCSPRespBuilder = new OCSPRespBuilder();
        byte[] byArray3 = oCSPRespBuilder.build(0, basicOCSPResp).getEncoded();
    }
}

